<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Exception;
use App\Mail\WeeklyReportMail;

class SendWeeklyReport extends Command
{
    protected $signature = 'report:weekly';
    protected $description = 'Send weekly financial report to admins';

    public function handle()
    {
        $this->info("🟢 Starting weekly financial report...");

        try {

            // ============================
            // 1. LOG EACH EXECUTION (FILE)
            // ============================
            file_put_contents(
                storage_path('logs/weekly_report_runs.log'),
                now() . " - report:weekly executed\n",
                FILE_APPEND
            );

            Log::info('Weekly report command started at ' . now());

            // ============================
            // 2. FORCE DB RECONNECT
            // ============================
            DB::purge('mysql');
            DB::reconnect('mysql');

            // ============================
            // 3. WEEK RANGE
            // ============================
            $startDate = now()->startOfWeek()->toDateString();
            $endDate   = now()->endOfWeek()->toDateString();

            // ============================
            // 4. FINANCIAL METRICS
            // ============================
            $totalPaid = DB::table('payments')
                ->whereBetween('payment_date', [$startDate, $endDate])
                ->sum('paid_amount');

            $totalDue = DB::table('payments')
                ->whereBetween('payment_date', [$startDate, $endDate])
                ->sum('amount_due');

            $totalAmount = $totalPaid + $totalDue;

            // ============================
            // 5. USERS / LOANS METRICS
            // ============================
            $allUsers = DB::table('loans')
                ->distinct('user_id')
                ->count('user_id');

            $customersWithLoans = $allUsers;

            $fullyPaid = DB::table('loans')
                ->join('payments', 'loans.id', '=', 'payments.loan_id')
                ->select('loans.user_id')
                ->groupBy('loans.user_id')
                ->havingRaw('SUM(payments.paid_amount) >= SUM(loans.loan_required_amount)')
                ->count();

            // ============================
            // 6. CONSOLE OUTPUT
            // ============================
            $this->info("Weekly Financial Report");
            $this->info("Week: $startDate to $endDate");
            $this->line("Total Amount (money stack): {$totalAmount} Tshs");
            $this->line("Paid Amount (money in): {$totalPaid} Tshs");
            $this->line("Due Amount (money out): {$totalDue} Tshs");
            $this->line("All Users (customers with loans): {$allUsers} People");
            $this->line("Fully Paid Customers: {$fullyPaid} People");
            $this->line("Thanks,\nBomarent");

            // ============================
            // 7. FETCH ADMIN EMAILS
            // ============================
            $admins = DB::table('users')
                ->where('role', 'admin')
                ->whereNotNull('email')
                ->where('email', '!=', '')
                ->pluck('email');

            if ($admins->isEmpty()) {

                $this->warn("⚠️ No admin has an email. Weekly report not sent.");
                Log::warning('Weekly report NOT sent: no admin emails found');

            } else {

                foreach ($admins as $email) {
                    Mail::to($email)->send(new WeeklyReportMail([
                        'start_date' => $startDate,
                        'end_date' => $endDate,
                        'total_amount' => $totalAmount,
                        'paid_amount' => $totalPaid,
                        'due_amount' => $totalDue,
                        'all_users' => $allUsers,
                        'customers_with_loans' => $customersWithLoans,
                        'fully_paid' => $fullyPaid,
                    ]));
                }

                $this->info("✅ Weekly report emailed to admins.");
                Log::info('Weekly report successfully emailed to admins');

            }

        } catch (Exception $e) {

            $this->error("❌ Failed to generate weekly report: " . $e->getMessage());

            Log::error('Weekly report error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);

        }

        return Command::SUCCESS;
    }
}

