<?php

namespace App\Http\Controllers;

use App\Models\GpsDevice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class GpsDeviceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $devices = GpsDevice::all();
        $assignedDevices = GpsDevice::where('assignment_status', 'assigned')->get();
        $unassignedDevices = GpsDevice::where('assignment_status', 'unassigned')->get();
        $activeDevices = GpsDevice::where('activity_status', 'active')->get();
        $inactiveDevices = GpsDevice::where('activity_status', 'inactive')->get();
        $powerOnDevices = GpsDevice::where('power_status', 'ON')->get();
        $powerOffDevices = GpsDevice::where('power_status', 'OFF')->get();

        $deviceCount = [
            'total' => $devices->count(),
            'assigned' => $assignedDevices->count(),
            'unassigned' => $unassignedDevices->count(),
            'active' => $activeDevices->count(),
            'inactive' => $inactiveDevices->count(),
            'power_on' => $powerOnDevices->count(),
            'power_off' => $powerOffDevices->count(),
        ];

        $deviceStatus = [
            'assigned' => $assignedDevices,
            'unassigned' => $unassignedDevices,
            'active' => $activeDevices,
            'inactive' => $inactiveDevices,
            'power_on' => $powerOnDevices,
            'power_off' => $powerOffDevices,
        ];

        return view("gps-devices.gps-devices", [
            "devices"=> $devices,
            'deviceCount' => $deviceCount,
            'deviceStatus' => $deviceStatus,
        ]);
    }

    public function assignGpsDevice()
    {
        $devices = GpsDevice::all();
        return view("gps-devices.assign-gps-device", [
            'devices' => $devices,
        ]);
    }

    public function updatePowerStatus(Request $request)
    {
        $device = GpsDevice::where('device_id', $request->device_id)->firstOrFail();

        $request->validate([
            'power_status' => 'required|in:ON,OFF',
            'power_status_notes' => 'nullable|string|max:255',
        ]);

        $device->update([
            'power_status' => $request->power_status,
            'power_status_updated_at' => now(),
            'power_status_notes' => $request->power_status_notes,
        ]);

        try {
            $this->sendPowerCommand($device->device_id, $request->power_status);
        } catch (\Exception $e) {
            Log::error('Failed to send power command to device ' . $device->device_id . ': ' . $e->getMessage());
            return response()->json(['error' => 'Failed to send command to device'], 500);
        }

        return response()->json([
            'message' => 'Power status updated successfully',
            'device' => $device,
        ]);
    }

    /**
     * Receive GPS data via API
     */
    public function receiveGps(Request $request)
    {
        $request->validate([
            'device_id' => 'required|string',
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);

        try {
            // Update existing record or insert if new
            DB::table('device_locations')->updateOrInsert(
                ['device_id' => $request->device_id],
                [
                    'latitude' => $request->latitude,
                    'longitude' => $request->longitude,
                    'updated_at' => now(),
                    'created_at' => now(),
                ]
            );

            return response()->json([
                'status' => 'success',
                'message' => 'GPS data saved successfully'
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to save GPS data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getPowerStatus(string $device_id)
    {
        $device = GpsDevice::where('device_id', $device_id)->firstOrFail();

        return response()->json([
            'device_id' => $device->device_id,
            'power_status' => $device->power_status,
            'power_status_updated_at' => $device->power_status_updated_at,
            'power_status_notes' => $device->power_status_notes,
        ]);
    }

    protected function sendPowerCommand(string $device_id, string $status)
    {
        Log::info("Sent $status command to device $device_id");
    }

    public function create()
    {
        //
    }

    public function store(Request $request)
    {
        //
    }

    public function show(GpsDevice $gpsDevice)
    {
        return view('gps-devices.show-gps-device', [
            'device' => $gpsDevice,
            'device_id' => $gpsDevice->device_id,
        ]);
    }

    public function edit(GpsDevice $gpsDevice)
    {
        //
    }

    public function update(Request $request, GpsDevice $gpsDevice)
    {
        //
    }

    public function destroy(GpsDevice $gpsDevice)
    {
        //
    }
}

