<?php

namespace App\Livewire;

use Carbon\Carbon;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;
use App\Models\Loan;
use App\Models\Payment;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Http;
use App\Jobs\InitiatePaymentJob;
use Illuminate\Pagination\LengthAwarePaginator;
use App\Jobs\SendSmsJob;
use Illuminate\Support\Str;

class LoanTableComponent extends Component
{
    use WithPagination, WithFileUploads;

    public $search = '';
    public $sortField = '';
    public $sortDirection = 'asc';
    public $selectedLoanId;
    public $phoneNumber;
    public $bankName;
    public $provider;
    public $paymentAmount;
    public $paymentMethod = 'cash';
    public $receipt;
    public $paymentDate; // 🔹 Added property for payment date
    public $showPaymentModal = false;
    public $isLoading = false;

    protected $updatesQueryString = ['search', 'sortField', 'sortDirection'];

    protected $listeners = [
        'confirm-send-reminder' => 'sendReminder',
        'deleteItemConfirmed' => 'deleteItemConfirmed'
    ];

    public function rules()
    {
        return [
            'paymentAmount' => 'required|numeric|min:1000',
            'paymentMethod' => 'required|in:cash,mobile_money,bank',
            'phoneNumber' => ['required_if:paymentMethod,mobile_money', 'string', 'regex:/^(\+255|0)[0-9]{9}$/'],
            'provider' => 'required_if:paymentMethod,mobile_money|in:Mpesa,TigoPesa,AirtelMoney,HaloPesa',
            'bankName' => 'required_if:paymentMethod,bank',
            'paymentDate' => 'required|date', // 🔹 validate payment date
            'receipt' => 'nullable|file|mimes:pdf,png,jpg,jpeg|max:10240',
        ];
    }

    public function openPaymentModal($loanId)
    {
        $this->selectedLoanId = $loanId;
        $this->showPaymentModal = true;
        $this->reset(['paymentMethod', 'paymentAmount', 'phoneNumber', 'provider', 'receipt', 'isLoading', 'paymentDate']);
        $this->paymentMethod = 'cash';
        $loan = Loan::find($loanId);
        if ($loan && $loan->user) {
            $this->phoneNumber = $loan->user->phone_number;
            $this->provider = 'Mpesa';
        }
        Log::info('Opened payment modal', ['loan_id' => $loanId]);
    }

    public function closePaymentModal()
    {
        $this->showPaymentModal = false;
        $this->reset(['paymentMethod', 'paymentAmount', 'phoneNumber', 'provider', 'receipt', 'selectedLoanId', 'isLoading', 'paymentDate']);
        $this->resetValidation();
        Log::info('Closed payment modal');
    }

    public function confirmDelete($model, $id)
    {
        $this->dispatch('confirm-delete', [
            'model' => $model,
            'id' => $id,
            'message' => 'Are you sure you want to delete this item?'
        ]);
    }

    public function deleteItemConfirmed($model, $id)
    {
        if ($model === 'loan') {
            $item = Loan::find($id);
            $itemName = 'Loan';
            if ($item && $item->status === 'ongoing') {
                session()->flash('error', "Cannot delete an ongoing loan!");
                $this->dispatch('notify', [
                    'type' => 'error',
                    'message' => "Cannot delete an ongoing loan!"
                ]);
                return;
            }
        } elseif ($model === 'payment') {
            $item = Payment::find($id);
            $itemName = 'Payment';
        } else {
            session()->flash('error', 'Unknown item type!');
            $this->dispatch('notify', [
                'type' => 'error',
                'message' => 'Unknown item type!'
            ]);
            return;
        }

        if ($item) {
            $item->delete();
            session()->flash('success', "$itemName deleted successfully!");
            $this->dispatch('notify', [
                'type' => 'success',
                'message' => "$itemName deleted successfully!"
            ]);
        } else {
            session()->flash('error', "$itemName not found!");
            $this->dispatch('notify', [
                'type' => 'error',
                'message' => "$itemName not found!"
            ]);
        }
    }

    public function addPayment()
    {
        $this->validate();
        $this->isLoading = true;
        $this->dispatch('payment-processing');

        Log::info('Adding payment', [
            'loan_id' => $this->selectedLoanId,
            'amount' => $this->paymentAmount,
            'phone_number' => $this->phoneNumber,
            'provider' => $this->provider,
            'payment_date' => $this->paymentDate,
        ]);

        try {
            $loan = Loan::findOrFail($this->selectedLoanId);
            if ($loan->status !== 'approved') {
                throw new \Exception('Payments can only be added for approved loans.');
            }

            $parsedDate = Carbon::parse($this->paymentDate)->format('Y-m-d'); // 🔹 use user input date

            if ($this->paymentMethod === 'mobile_money') {
                $normalizedPhoneNumber = $this->normalizePhoneNumber($this->phoneNumber);
                InitiatePaymentJob::dispatch($loan->id, $this->paymentAmount, $normalizedPhoneNumber, $this->provider);

                $transactionId = 'MM_' . $loan->id . '_' . Str::uuid();

                Payment::create([
                    'loan_id' => $loan->id,
                    'user_id' => $loan->user->id,
                    'paid_amount' => $this->paymentAmount,
                    'payment_date' => $parsedDate,
                    'transaction_id' => $transactionId,
                    'external_id' => $transactionId,
                    'status' => 'pending',
                    'job_status' => 'processing',
                    'payment_method' => 'mobile_money',
                    'provider' => $this->provider,
                    'receipt_path' => null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                $recipients = [$normalizedPhoneNumber];
                $message = "Payment of TZS {$this->paymentAmount} for Loan #{$loan->id} has been initiated.";
                SendSmsJob::dispatch($recipients, $message, $loan->id);

            } elseif ($this->paymentMethod === 'cash') {
                $userName = str_replace(' ', '_', strtolower($loan->user->name ?? 'user_' . $loan->id));
                $timestamp = now()->format('YmdHis');
                Storage::disk('public')->makeDirectory('cash-receipts');

                $path = null;
                if ($this->receipt) {
                    $extension = $this->receipt->getClientOriginalExtension();
                    $fileName = "{$userName}_receipt_{$timestamp}.{$extension}";
                    $path = $this->receipt->storeAs('cash-receipts', $fileName, 'public');
                    if (!Storage::disk('public')->exists($path)) {
                        throw new \Exception("Failed to store receipt: {$fileName}");
                    }
                }

                $transactionId = 'CASH_' . $loan->id . '_' . Str::uuid();
                Payment::create([
                    'loan_id' => $loan->id,
                    'user_id' => $loan->user->id,
                    'paid_amount' => $this->paymentAmount,
                    'payment_date' => $parsedDate,
                    'transaction_id' => $transactionId,
                    'external_id' => $transactionId,
                    'status' => 'pending',
                    'job_status' => 'completed',
                    'payment_method' => 'cash',
                    'provider' => 'Cash',
                    'receipt_path' => $path,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                $recipients = [$this->normalizePhoneNumber($loan->user->phone_number)];
                $message = "Cash payment of TZS {$this->paymentAmount} for Loan #{$loan->id} recorded, pending verification.";
                SendSmsJob::dispatch($recipients, $message, $loan->id);

            } elseif ($this->paymentMethod === 'bank') {
                $transactionId = 'BANK_' . $loan->id . '_' . Str::uuid();
                $bankProvider = $this->provider;

                Payment::create([
                    'loan_id' => $loan->id,
                    'user_id' => $loan->user->id,
                    'paid_amount' => $this->paymentAmount,
                    'payment_date' => $parsedDate,
                    'transaction_id' => $transactionId,
                    'external_id' => $transactionId,
                    'status' => 'pending',
                    'job_status' => 'completed',
                    'payment_method' => 'bank',
                    'provider' => $bankProvider,
                    'receipt_path' => null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                $recipients = [$this->normalizePhoneNumber($loan->user->phone_number)];
                $message = "Bank payment of TZS {$this->paymentAmount} for Loan #{$loan->id} recorded, pending verification.";
                SendSmsJob::dispatch($recipients, $message, $loan->id);
            }

            if (isset($loan->user->email)) {
                \Mail::to($loan->user->email)->send(new \App\Mail\PaymentNotificationMail($loan, $this->paymentAmount, $this->paymentMethod));
            }

            session()->flash('message', $this->paymentMethod === 'cash'
                ? 'Cash payment recorded successfully.'
                : 'Payment initiated successfully. You will be notified once processed.');
            $this->dispatch('notify', [
                'type' => 'success',
                'message' => session()->get('message'),
            ]);
            $this->closePaymentModal();
        } catch (\Exception $e) {
            Log::error('Failed to add payment: ' . $e->getMessage(), [
                'loan_id' => $this->selectedLoanId,
                'trace' => $e->getTraceAsString(),
            ]);
            session()->flash('error', 'Failed to add payment: ' . $e->getMessage());
            $this->dispatch('notify', [
                'type' => 'error',
                'message' => 'Failed to add payment: ' . $e->getMessage(),
            ]);
        } finally {
            $this->isLoading = false;
            $this->dispatch('payment-processed');
        }
    }

    protected function normalizePhoneNumber(string $phoneNumber): string
    {
        $phoneNumber = preg_replace('/\D/', '', $phoneNumber);
        if (preg_match('/^0(6|7|8)\d{8}$/', $phoneNumber)) {
            return '+255' . substr($phoneNumber, 1);
        }
        if (preg_match('/^(?:\+?255)(6|7|8)\d{8}$/', $phoneNumber)) {
            return '+' . ltrim($phoneNumber, '+');
        }
        return $phoneNumber;
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function resetSorting()
    {
        $this->sortField = '';
        $this->sortDirection = 'asc';
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
        $this->resetPage();
    }

    public function render()
    {
        $payments = Payment::with('loan')->get();
        $user = Auth::user();
        $totalLoanAmount = Loan::sum('loan_required_amount');
        $users = User::with('loans.payments')->get();

        $nearEndLoans = Loan::whereNotNull('loan_end_date')
            ->whereDate('loan_end_date', '>', now())
            ->orderBy('loan_end_date', 'asc')
            ->take(12)
            ->with('user')
            ->get();

        $loans = Loan::with(['user', 'payments'])
            ->where('status', 'approved')
            ->when($this->search, function ($query) {
                $query->where('applicant_name', 'like', "%{$this->search}%");
            })
            ->get()
            ->map(function ($loan) {
                $paid = $loan->payments->sum('paid_amount');
                $remaining = $loan->loan_required_amount - $paid;
                $loan->amount_paid = $paid;
                $loan->amount_remaining = $remaining;
                $loan->days_remaining = $loan->loan_end_date
                    ? now()->diffInDays($loan->loan_end_date, false)
                    : null;
                return $loan;
            });

        if ($this->sortField) {
            $loans = $loans->sortBy($this->sortField, SORT_REGULAR, $this->sortDirection === 'desc');
        }

        $paginatedLoans = $this->paginateCollection($loans, 10);
        $today = Carbon::now();

        return view('livewire.loan-table-component', [
            'loans' => $paginatedLoans,
            'payments' => $payments,
            'user' => $user,
            'users' => $users,
            'totalLoanAmount' => $totalLoanAmount,
            'nearEndLoans' => $nearEndLoans,
            'today' => $today,
        ]);
    }

    public function paginateCollection($items, $perPage = 10)
    {
        $page = request()->get('page', 1);
        $items = collect($items);
        $pagedItems = $items->slice(($page - 1) * $perPage, $perPage)->values();
        return new LengthAwarePaginator($pagedItems, $items->count(), $perPage, $page, [
            'path' => request()->url(),
            'query' => request()->query(),
        ]);
    }

    public function deleteLoan($loanId)
    {
        $this->deleteItemConfirmed('loan', $loanId);
    }

    public function sendReminder($loanId)
    {
        $loan = Loan::with(['user', 'payments'])->findOrFail($loanId);
        try {
            $phoneNumber = $this->normalizePhoneNumber($loan->user->phone_number);
            $nextDueDate = $loan->payments->max('payment_date')
                ? Carbon::parse($loan->payments->max('payment_date'))->addWeek()
                : Carbon::parse($loan->created_at)->addWeek();

            $formattedDate = $nextDueDate->format('d/m/Y');
            $message = "Habari {$loan->user->first_name}, Kumbukumbu ya malipo yako ijayo ni tarehe $formattedDate. Tafadhali hakikisha unalipa kwa wakati.";

            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/x-www-form-urlencoded',
                'apiKey' => config('services.africastalking.api_key'),
            ])->asForm()->post('https://api.africastalking.com/version1/messaging', [
                'username' => config('services.africastalking.username'),
                'to' => $phoneNumber,
                'from' => 'NK CNG',
                'message' => $message,
                'enqueue' => 1,
            ]);

            if ($response->successful()) {
                Log::info('SMS sent successfully to ' . $loan->user->first_name . ' at ' . $phoneNumber);
                $this->dispatch('notify', [
                    'type' => 'success',
                    'message' => "Ujumbe umetumwa kwa {$loan->user->first_name} ({$phoneNumber})"
                ]);
                return true;
            }
            throw new \Exception("SMS API request failed: " . $response->body());
        } catch (\Exception $e) {
            Log::error("sendReminder failed: " . $e->getMessage());
            $this->dispatch('notify', [
                'type' => 'error',
                'message' => 'Tatizo limetokea wakati wa kutuma ujumbe: ' . $e->getMessage()
            ]);
        }
    }
}
